import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.PriorityQueue;

public class HospitalSimulation {
    private final PriorityQueue<Patient> triageQueue;

    public HospitalSimulation() {
        // Create a PriorityQueue sorting patients from highest severity to lowest
        this.triageQueue = new PriorityQueue<>(Comparator.comparingInt(Patient::getSeverity).reversed());
    }

    // Add patients to the queue
    public void admitPatients(Patient[] patients) {
        System.out.println("🚑 Admitting patients...");
        for (Patient patient : patients) {
            triageQueue.add(patient);
            System.out.println("✅ Admitted: " + patient);
        }
    }

    // Treat patients based on their severity
    public void treatPatients() {
        System.out.println("\n🏥 Starting treatment...");
        while (!triageQueue.isEmpty()) {
            Patient currentPatient = triageQueue.poll(); // Remove patient with the highest severity
            System.out.println("👨‍⚕️ Treating: " + currentPatient);
            simulateTreatment();
            System.out.println("✅ " + currentPatient.getName() + " has been treated.\n");

            // After treating a patient, increase the severity of waiting patients
            worsenWaitingPatients();
        }
        System.out.println("🎉 All patients have been treated or deceased!");
    }

    // Increase severity for waiting patients, handle deaths
    private void worsenWaitingPatients() {
        if (triageQueue.isEmpty()) {
            return;
        }

        List<Patient> updatedPatients = new ArrayList<>();
        List<Patient> deceasedPatients = new ArrayList<>();

        while (!triageQueue.isEmpty()) {
            Patient patient = triageQueue.poll();
            if (patient.getSeverity() >= 5) {
                patient.increaseSeverity();
                if (patient.getSeverity() > 10) {
                    deceasedPatients.add(patient); // Patient dies
                    continue;
                }
            }
            updatedPatients.add(patient);
        }

        // Rebuild the queue with updated patients
        triageQueue.addAll(updatedPatients);

        // Report deaths
        for (Patient deceased : deceasedPatients) {
            System.out.println("⚰️ Patient " + deceased.getName() + " has died while waiting (severity exceeded 10).");
        }
    }

    // Simulate the time taken to treat a patient
    private void simulateTreatment() {
        try {
            Thread.sleep(500); // Simulate short delay (500ms)
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    // Entry point
    public static void main(String[] args) {
        // Create hospital simulation instance
        HospitalSimulation hospital = new HospitalSimulation();

        // Create 10 patients arriving at the hospital
        Patient[] patients = {
                new Patient("Alena"),
                new Patient("Beáta"),
                new Patient("Ctibor"),
                new Patient("Daniel"),
                new Patient("Elena"),
                new Patient("František"),
                new Patient("Gregor"),
                new Patient("Helena"),
                new Patient("Igor"),
                new Patient("Jana")
        };

        // Simulate admitting and treating patients
        hospital.admitPatients(patients);
        hospital.treatPatients();
    }
}
